package ga.core.algorithm.interactive;

import ga.core.GA;
import ga.core.algorithm.util.RandomSingleton;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.IMutationOp;
import ga.core.individual.IClusterableIndividual;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IClusterPopulation;
import ga.core.individual.population.IPopulation;
import ga.core.logging.IGALogger;
import ga.core.selection.ISelector;
import ga.core.validation.IValidator;

import java.util.logging.Logger;

/**
 * Steady-State simple interactive genetic algorithm.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class SIGA<T extends IIndividual<T>> extends AbstractSIGA<T> {
  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(SIGA.class.getName());

  private static final int CRITICAL_WHILE_ITERATIONS = 10000;

  /**
   * Creates a new steady state SIGA.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SIGA(final IPopulation<T> population,
      final IInteractiveFitnessEvaluator<T> evaluator,
      final ISelector<T> selector, final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final boolean useEliteStrategy) {
    this(population, evaluator, selector, mutateOperator, crossoverOperator,
        null, useEliteStrategy, null);
  }

  /**
   * Creates a new steady state SIGA.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param validator
   *          The validator
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * @param gaLogger
   *          The ga logger.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SIGA(final IPopulation<T> population,
      final IInteractiveFitnessEvaluator<T> evaluator,
      final ISelector<T> selector, final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final IValidator<T> validator,
      final boolean useEliteStrategy, final IGALogger<T> gaLogger) {
    super(population, evaluator, selector, mutateOperator, crossoverOperator,
        validator, useEliteStrategy, gaLogger);

    getContext().put(GA.KEY_EVALUATED_INDIVIDUALS_MIN_COUNT, 4);
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void step() {
    checkThread();

    incGeneration();

    IndividualList<T> selectedIndividuals;

    // SELECT
    selectedIndividuals = getSelector().select(getPopulation());

    if (getGALogger() != null) {
      getGALogger().individualsSelected(getGeneration(), selectedIndividuals);
    }

    final T ind1 = selectedIndividuals.get(0);
    final T ind2 = selectedIndividuals.get(1);

    int whileCounter = 0;
    do {
      // CROSSOVER
      final IndividualList<T> newSelectedIndividuals = getCrossoverOp()
          .crossover(ind1, ind2, getContext());
      if (newSelectedIndividuals != null) {
        selectedIndividuals = newSelectedIndividuals;

        if (getGALogger() != null) {
          getGALogger().individualsCrossed(getGeneration(),
              newSelectedIndividuals);
        }

      } else {
        LOGGER.warning("Crossover operation returned null");
      }

      // since we do not know how much individuals are returned by the
      // crossover, mutate all
      for (int i = 0; i < selectedIndividuals.size(); i++) {
        final T ind = selectedIndividuals.get(i);
        // MUTATE
        final T mutatedIndividual = getMutationOp().mutate(ind, getContext());
        if (mutatedIndividual != null) {
          selectedIndividuals.set(i, mutatedIndividual);

        } else {
          LOGGER.warning("Mutation operation returned null");
        }
      }

      if (getGALogger() != null) {
        getGALogger().individualsMutated(getGeneration(), selectedIndividuals);
      }

      whileCounter++;

      if (whileCounter > CRITICAL_WHILE_ITERATIONS) {
        LOGGER
            .warning("Critical iterations exceeded. Endless loop? Individuals: "
                + ind1 + "   " + ind2);

        for (int i = 0; i < selectedIndividuals.size(); i++) {
          // do a extra mutate
          final T ind = selectedIndividuals.get(i);
          // MUTATE
          final T mutatedIndividual = getMutationOp().mutate(ind, getContext());
          if (mutatedIndividual != null) {
            selectedIndividuals.set(i, mutatedIndividual);

          } else {
            LOGGER.warning("Mutation operation returned null");
          }
        }
      }

    } while ((isValidate() && getValidator() != null && !selectedIndividuals
        .isValid(getValidator(), getContext()))
        || (!getPopulation().isAllowDuplicates() && getPopulation()
            .containsAny(selectedIndividuals))

    );

    // INSERT
    getSelector().insert(selectedIndividuals, getPopulation(),
        isUseEliteStrategy());

    if (getPopulation() instanceof IClusterPopulation) {
      ((IClusterPopulation) getPopulation()).doClustering();
    }

    if (getGALogger() != null) {
      getGALogger().individualsInserted(getGeneration(), selectedIndividuals,
          getPopulation());
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see ga.core.algorithm.interactive.ISIGA#newIndividualRequested()
   * 
   * @since 06.01.2012
   * 
   * @author Stephan Dreyer
   */
  @Override
  public void newIndividualRequested() {
    checkThread();

    T ind = null;

    if (!getPopulation().isEmpty()) {
      final IndividualList<T> list = getPopulation()
          .getUnevaluatedIndividuals();

      LOGGER.info("New individual from population");

      if (list.size() > 0) {
        do {
          ind = list.get(RandomSingleton.getRandom().nextInt(list.size()));
        } while (getEvaluatingIndividuals().contains(ind));

        LOGGER.info("Evaluate " + ind);
      } else {
        do {
          ind = getPopulation().getRandomIndividualForEvaluation();
        } while (getEvaluatingIndividuals().contains(ind));
      }
    } else {
      LOGGER.info("Your Population is too small");
      getPopulation().initRandomly(getValidator(), getContext());

      if (getGALogger() != null) {
        getGALogger().populationInitiated(getGeneration(), getPopulation());
      }

      do {
        ind = getPopulation().getRandomIndividualForEvaluation();
      } while (getEvaluatingIndividuals().contains(ind));
    }

    if (getGALogger() != null) {
      getGALogger().individualSelectedForEvaluation(getGeneration(), ind);
    }

    getEvaluatingIndividuals().add(ind);
    getEvaluator().evaluate(ind);
  }

  /*
   * (non-Javadoc)
   * 
   * @see ga.core.algorithm.interactive.ISIGA#individualEvaluated(T)
   * 
   * @since 06.01.2012
   * 
   * @author Stephan Dreyer
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  @Override
  public void individualEvaluated(final T ind) {
    checkThread();

    getEvaluatingIndividuals().remove(ind);

    if (getPopulation() instanceof IClusterPopulation) {
      if (ind instanceof IClusterableIndividual) {
        ((IClusterPopulation) getPopulation())
            .assignFitness((IClusterableIndividual) ind);
      } else {
        throw new RuntimeException(ind + " is not clusterable");
      }
    }

    if (getGALogger() != null) {
      getGALogger().individualEvaluated(getGeneration(), ind);
    }

    if (getPopulation().getEvaluatedIndividualCount() >= (Integer) getContext()
        .get(GA.KEY_EVALUATED_INDIVIDUALS_MIN_COUNT)) {
      step();
    }
  }
}
